TerrainHelper = {}

function TerrainHelper:raiseGround(workArea, callback)
    self.callback = callback
    local layer = -1

    self:createAdditiveDeformation(workArea.heightChangeAmount)
    self:createSmoothingDeformation(0.5)
    self:createPaintingDeformation()

    local additiveDeformation = self.additiveDeformation

    local paintDeformation = self.paintDeformation
    local paintingLayer = self:getLayer(workArea)

    local smoothingDeformation = self.smoothingDeformation
    local smoothStrength = 1
    if workArea.groundTypeText == 'asphalt' then
        smoothStrength = 0.8
    end if workArea.groundTypeText == 'gravel' then
        smoothStrength = 0.4
    end

    for _, brushPoint in pairs(self:getBrushPoints(workArea)) do
        additiveDeformation:addSoftSquareBrush(brushPoint.x, brushPoint.z, brushPoint.radius, workArea.hardness * 0.95, workArea.strength, layer)
        paintDeformation:addSoftSquareBrush(brushPoint.x, brushPoint.z, brushPoint.radius + 0.1 , workArea.hardness * 0.95, workArea.strength, paintingLayer)
        smoothingDeformation:addSoftSquareBrush(brushPoint.x, brushPoint.z, brushPoint.radius + 0.3 , workArea.hardness * 0.95, smoothStrength, layer)
    end

    additiveDeformation:setOutsideAreaConstraints(2, math.rad(65), math.rad(65))
    additiveDeformation:setBlockedAreaMaxDisplacement(0)
    additiveDeformation:setDynamicObjectCollisionMask(0)
    additiveDeformation:setDynamicObjectMaxDisplacement(0)

    smoothingDeformation:setOutsideAreaConstraints(2, math.rad(65), math.rad(65))
    smoothingDeformation:setBlockedAreaMaxDisplacement(0)
    smoothingDeformation:setDynamicObjectCollisionMask(0)
    smoothingDeformation:setDynamicObjectMaxDisplacement(0)


    local x, _, z = getWorldTranslation(workArea.start)
    local x1, _, z1 = getWorldTranslation(workArea.width)
    local x2, _, z2 = getWorldTranslation(workArea.height)
    FSDensityMapUtil.removeFieldArea(x, z, x1, z1, x2, z2, false)
    FSDensityMapUtil.removeWeedArea(x, z, x1, z1, x2, z2)
    FSDensityMapUtil.removeStoneArea(x, z, x1, z1, x2, z2)
    DensityMapHeightUtil.clearArea(x, z, x1, z1, x2, z2)
    FSDensityMapUtil.clearDecoArea(x, z, x1, z1, x2, z2)
    FSDensityMapUtil.eraseTireTrack(x, z, x1, z1, x2, z2)

    additiveDeformation:apply(false, 'onAdditiveDeformationCallback', self)
    paintDeformation:apply(false, 'onPaintingDeformationCallback', self)
    smoothingDeformation:apply(false, 'onSmoothingDeformationCallback', self)
end

function TerrainHelper:onAdditiveDeformationCallback(code, volume)
    if code == TerrainDeformation.STATE_SUCCESS then
        if self.callback then
            self.callback(volume)
        end
    else
        Logging.error('onAdditiveDeformationCallback error code: %s', code)
    end
end

function TerrainHelper:getBrushPoints(workArea, squareSize)
    if squareSize == nil then
        squareSize = 0.4
    end
    local startX, _, startZ = getWorldTranslation(workArea.start)
    local endX, _, endZ = getWorldTranslation(workArea.width)
    local points = {}
    local dx = endX - startX
    local dz = endZ - startZ
    local distance = math.sqrt(dx * dx + dz * dz)
    local numSquares = math.ceil(distance / squareSize)
    local remainingDistance = distance - numSquares * squareSize

    for i = 0, numSquares - 1 do
        local x = startX + (dx / distance) * (i * squareSize)
        local z = startZ + (dz / distance) * (i * squareSize)
        table.insert(points, {x = x, z = z, radius = squareSize})
    end

    -- Add the last point, which is the end point
    table.insert(points, {x = endX, z = endZ, radius = squareSize})

    return points
end

function TerrainHelper:getLayer(workArea)
    return g_groundTypeManager:getTerrainLayerByType(workArea.groundTypeText)
end

function TerrainHelper:createBaseDeformation()
    return TerrainDeformation.new(g_currentMission.terrainRootNode)
end

function TerrainHelper:createAdditiveDeformation(heightChangeAmount)
    self.additiveDeformation = self:createBaseDeformation()

    self.additiveDeformation:enableAdditiveDeformationMode()
    self.additiveDeformation:setAdditiveHeightChangeAmount(heightChangeAmount)
end

function TerrainHelper:createSmoothingDeformation(heightChangeAmount)
    self.smoothingDeformation = self:createBaseDeformation()

    self.smoothingDeformation:setAdditiveHeightChangeAmount(heightChangeAmount)
    self.smoothingDeformation:enableSmoothingMode()
end

function TerrainHelper:createFlattenDeformation(targetY, heightChangeAmount)
    self.flattenDeformation = self:createBaseDeformation()

    if heightChangeAmount == nil then
        heightChangeAmount = 1
    end

    self.flattenDeformation:setAdditiveHeightChangeAmount(heightChangeAmount)
    self.flattenDeformation:setHeightTarget(targetY, targetY, 0, 1, 0, -targetY)
    self.flattenDeformation:enableSetDeformationMode()
end

function TerrainHelper:createPaintingDeformation()
    self.paintDeformation = self:createBaseDeformation()

    self.paintDeformation:enablePaintingMode()
end

return TerrainHelper