local modName = g_currentModName
RoadCraftTrailer = {}
RoadCraftTrailer.SPEC_TABLE_NAME = "spec_"..modName..".RoadCraftTrailer"
RoadCraftTrailer.STEP_SIZE = 0.1
RoadCraftTrailer.MIN_STRENGTH = 0.1
RoadCraftTrailer.MAX_STRENGTH = 1
RoadCraftTrailer.GROUNDTYPE_STEP_SIZE = 1
RoadCraftTrailer.GROUNDTYPE_MIN = 1
RoadCraftTrailer.GROUNDTYPE_MAX = 2
RoadCraftTrailer.fillUnitIndex = 1
RoadCraftTrailer.NUM_BITS = 8
RoadCraftTrailer.VOLUME_FACTOR = {
    [1] = 200,
    [2] = 100
}
RoadCraftTrailer.GROUNDTYPES = {
    [1] = "asphalt",
    [2] = "gravel"
}

function RoadCraftTrailer.prerequisitesPresent(specializations)
    local check success = false
    if SpecializationUtil.hasSpecialization(Attachable, specializations) and SpecializationUtil.hasSpecialization(TurnOnVehicle, specializations) then
        return true
    else
        return false
    end 
end

function RoadCraftTrailer.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", RoadCraftTrailer)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", RoadCraftTrailer)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", RoadCraftTrailer)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", RoadCraftTrailer)
    SpecializationUtil.registerEventListener(vehicleType, "onTurnedOn", RoadCraftTrailer)
    SpecializationUtil.registerEventListener(vehicleType, "onTurnedOff", RoadCraftTrailer)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", RoadCraftTrailer)
end

function RoadCraftTrailer.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "processWorkAreaPlaceholder",RoadCraftTrailer.processWorkAreaPlaceholder)
    SpecializationUtil.registerFunction(vehicleType, "setStrengthFactor",RoadCraftTrailer.setStrengthFactor)
    SpecializationUtil.registerFunction(vehicleType, "getStrengthFactor",RoadCraftTrailer.getStrengthFactor)
    SpecializationUtil.registerFunction(vehicleType, "setGroundType",RoadCraftTrailer.setGroundType)
    SpecializationUtil.registerFunction(vehicleType, "getGroundType",RoadCraftTrailer.getGroundType)
    SpecializationUtil.registerFunction(vehicleType, "onAdditiveDeformationCallback",RoadCraftTrailer.onAdditiveDeformationCallback)
end

function RoadCraftTrailer.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "doCheckSpeedLimit",RoadCraftTrailer.doCheckSpeedLimit)
end

function RoadCraftTrailer.initSpecialization()
    g_workAreaTypeManager:addWorkAreaType("roadCraftTrailer", false)
	local schema = Vehicle.xmlSchema
	schema:setXMLSpecializationType("RoadCraftTrailer")
	schema:register(XMLValueType.FLOAT,"vehicle.RoadCraftTrailer#defaultStrength", "Default strength.")
	schema:register(XMLValueType.INT,"vehicle.RoadCraftTrailer#defaultGroundType", "Default Ground Type to apply")
schema:register(XMLValueType.FLOAT,"vehicle.RoadCraftTrailer#unloadInfoIndex", "IDK")
    schema:register(XMLValueType.STRING, "vehicle.RoadCraftTrailer.startWorkAnimation#name", "startWorkAnimation name")
    schema:register(XMLValueType.FLOAT, "vehicle.RoadCraftTrailer.startWorkAnimation#speedScale", "startWorkAnimation speed scale")
    schema:register(XMLValueType.FLOAT, "vehicle.RoadCraftTrailer.startWorkAnimation#time", "startWorkAnimation time")
    schema:register(XMLValueType.STRING, "vehicle.RoadCraftTrailer.stopWorkAnimation#name", "stopWorkAnimation name")
    schema:register(XMLValueType.FLOAT, "vehicle.RoadCraftTrailer.stopWorkAnimation#speedScale", "stopWorkAnimation speed scale")
    schema:register(XMLValueType.FLOAT, "vehicle.RoadCraftTrailer.stopWorkAnimation#time", "stopWorkAnimation time")
    schema:register(XMLValueType.FLOAT, "vehicle.RoadCraftTrailer#heightChangeAmount", "height change amount")
	schema:setXMLSpecializationType()
end

function RoadCraftTrailer:onLoad(savegame)
	Logging.info(modName..": onLoad", modName)
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
	
    if spec == nil then
        Logging.warning(modName..": RoadCraftTrailer spec missing")
        spec = {}
    end

    -- fill type indices
    spec.stoneIndex  = g_fillTypeManager:getFillTypeIndexByName("STONE")

    spec.unloadInfoIndex = self.xmlFile:getValue("vehicle.RoadCraftTrailer#unloadInfoIndex", 1)

    if not self.xmlFile:hasProperty("vehicle.RoadCraftTrailer") then
        Logging.xmlWarning(self.xmlFile, "Missing RoadCraftTrailer Tag")
    end

    -- load workArea
    spec.workArea = self:getWorkAreaByIndex(1)
    if spec.workArea == nil then
        Logging.error("workArea missing")
    end

    -- strengths: raise & smooth
    spec.strengthFactor   = self.xmlFile:getValue("vehicle.RoadCraftTrailer#defaultStrength", 0.5)
    spec.workArea.groundType = self.xmlFile:getValue("vehicle.RoadCraftTrailer#defaultGroundType", 1)
    spec.workArea.heightChangeAmount = self.xmlFile:getValue("vehicle.RoadCraftTrailer#heightChangeAmount", 0.0025)

    spec.workArea.groundTypeText = self:getGroundType()

    spec.workArea.hardness = 1.0

    spec.startWorkAnimation = {}
    spec.startWorkAnimation.name = self.xmlFile:getValue("vehicle.RoadCraftTrailer.startWorkAnimation#name")
    spec.startWorkAnimation.speedScale = self.xmlFile:getValue("vehicle.RoadCraftTrailer.startWorkAnimation#speedScale", 1)
    spec.startWorkAnimation.time = self.xmlFile:getValue("vehicle.RoadCraftTrailer.startWorkAnimation#time", 1)

    spec.stopWorkAnimation = {}
    spec.stopWorkAnimation.name = self.xmlFile:getValue("vehicle.RoadCraftTrailer.stopWorkAnimation#name")
    spec.stopWorkAnimation.speedScale = self.xmlFile:getValue("vehicle.RoadCraftTrailer.stopWorkAnimation#speedScale", 1)
    spec.stopWorkAnimation.time = self.xmlFile:getValue("vehicle.RoadCraftTrailer.stopWorkAnimation#time", 1)
end

function RoadCraftTrailer:onRegisterActionEvents(isActiveForInput,isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
        self:clearActionEventsTable(spec.actionEvents)
        if isActiveForInputIgnoreSelection then
            local _, actionEventId = self:addActionEvent(spec.actionEvents,InputAction.CHANGE_STRENGTH, self,RoadCraftTrailer.actionEventChangeStrength, false, true,false, true, nil)
            g_inputBinding:setActionEventText(actionEventId,string.format(g_i18n:getText("input_CHANGE_STRENGTH"),self:getStrengthFactor()*100 + 0.1))
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
            local _, actionEventId = self:addActionEvent(spec.actionEvents,InputAction.CHANGE_GROUND_TYPE, self,RoadCraftTrailer.actionEventChangeGroundType, false, true,false, true, nil)
            g_inputBinding:setActionEventText(actionEventId,string.format(g_i18n:getText("input_CHANGE_GROUND_TYPE"),self:getGroundType()))
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
        end
    end                                                                                                                
end

function RoadCraftTrailer.actionEventChangeStrength(self, actionName, inputValue, callbackState, isAnalog)
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    local step = inputValue * RoadCraftTrailer.STEP_SIZE
    local newFactor = spec.strengthFactor + step
    self:setStrengthFactor(newFactor, false)
end

function RoadCraftTrailer.actionEventChangeGroundType(self, actionName, inputValue, callbackState, isAnalog)
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    local groundType = spec.workArea.groundType
    local step = inputValue * RoadCraftTrailer.GROUNDTYPE_STEP_SIZE
    if groundType == nil then
        Logging.warning(modName..": groundType is nil, setting to min")
        groundType = RoadCraftTrailer.GROUNDTYPE_MIN
    end
    groundType = groundType + step
    if groundType > RoadCraftTrailer.GROUNDTYPE_MAX then
        groundType = RoadCraftTrailer.GROUNDTYPE_MIN
    end
    if groundType < RoadCraftTrailer.GROUNDTYPE_MIN then
        groundType = RoadCraftTrailer.GROUNDTYPE_MAX
    end
    self:setGroundType(groundType)
end

function RoadCraftTrailer:setGroundType(groundType, noEventSend)
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    if spec.workArea.groundType ~= groundType then
        spec.workArea.groundType = groundType
        RoadCraftChangeGroundTypeEvent.sendEvent(self, groundType, noEventSend)
        local actionEvent = spec.actionEvents[InputAction.CHANGE_GROUND_TYPE]
        if actionEvent ~= nil then
            g_inputBinding:setActionEventText(actionEvent.actionEventId,string.format(g_i18n:getText("input_CHANGE_GROUND_TYPE"),self:getGroundType()))
        end
    end
end

function RoadCraftTrailer:setStrengthFactor(factor, noEventSend)
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    factor = math.min(RoadCraftTrailer.MAX_STRENGTH, math.max(RoadCraftTrailer.MIN_STRENGTH, factor))
    if math.abs(spec.strengthFactor - factor) > 0.0001 then
        spec.strengthFactor = factor
        RoadCraftChangeStrengthEvent.sendEvent(self, factor, noEventSend)
        local actionEvent = spec.actionEvents[InputAction.CHANGE_STRENGTH]
        if actionEvent ~= nil then
            g_inputBinding:setActionEventText(actionEvent.actionEventId,string.format(g_i18n:getText("input_CHANGE_STRENGTH"),self:getStrengthFactor()*100 + 0.1))
        end
    end
end

function RoadCraftTrailer:getGroundType()
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    return RoadCraftTrailer.GROUNDTYPES[spec.workArea.groundType]
end

function RoadCraftTrailer:getStrengthFactor()
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    return spec.strengthFactor
end

function RoadCraftTrailer:onTurnedOn()
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    if self:getFillUnitFillLevel(RoadCraftTrailer.fillUnitIndex) > 0 then
    else
        self:setIsTurnedOn(false)
    end
    if self.isClient then
        self:setAnimationTime(spec.startWorkAnimation.name, 0, true)
        self:playAnimation(spec.startWorkAnimation.name, spec.startWorkAnimation.speedScale, self:getAnimationTime(spec.startWorkAnimation.name))
    end
end

function RoadCraftTrailer:onTurnedOff()
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    if self.isClient then
        self:setAnimationTime(spec.stopWorkAnimation.name, 0, true)
        self:playAnimation(spec.stopWorkAnimation.name, spec.stopWorkAnimation.speedScale, self:getAnimationTime(spec.stopWorkAnimation.name))
    end
end

function RoadCraftTrailer:onUpdateTick(dt)
    if self.isServer then
        local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
        local speedKmh = self:getLastSpeed()
        if self:getIsTurnedOn() and speedKmh > 1 then

            local workArea = spec.workArea

            local fillLevel = self:getFillUnitFillLevel(RoadCraftTrailer.fillUnitIndex)
            if fillLevel > 0 then
                workArea.strength = self:getStrengthFactor()
                workArea.groundTypeText = self:getGroundType()

                TerrainHelper:raiseGround(workArea, function(volume) self:onAdditiveDeformationCallback(volume) end)
            else
                self:setIsTurnedOn(false)
            end
        end
    end
end

function RoadCraftTrailer:onAdditiveDeformationCallback(volume)
    local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
    volume = volume * RoadCraftTrailer.VOLUME_FACTOR[spec.workArea.groundType]
    self:addFillUnitFillLevel(self:getOwnerFarmId(), RoadCraftTrailer.fillUnitIndex, -volume, self:getFillUnitFillType(RoadCraftTrailer.fillUnitIndex), ToolType.UNDEFINED, self:getFillVolumeUnloadInfo(spec.unloadInfoIndex))
end

function RoadCraftTrailer:doCheckSpeedLimit(superFunc)
    if self:getIsTurnedOn() then
        return true
    end
    return superFunc(self)
end

function RoadCraftTrailer:onReadStream(streamId, connection)
	Logging.info(modName..": onReadStream", modName)
	local strength = RoadCraftTrailer.streamReadStrengthFactor(streamId)
    self:setStrengthFactor(strength, true)
end

function RoadCraftTrailer:onWriteStream(streamId, connection)
	Logging.info(modName..": onWriteStream", modName)
	local spec = self[RoadCraftTrailer.SPEC_TABLE_NAME]
	RoadCraftTrailer.streamWriteStrengthFactor(streamId, spec.strengthFactor)
end

function RoadCraftTrailer.streamWriteStrengthFactor(streamId, strengthFactor)
    streamWriteUIntN(streamId, math.floor((strengthFactor * 10) + 0.5),RoadCraftTrailer.NUM_BITS)
end

function RoadCraftTrailer.streamReadStrengthFactor(streamId)
    -- read the speed factor from the network stream as -- an integer
    local strength = streamReadUIntN(streamId, RoadCraftTrailer.NUM_BITS)
    local strengthFactor = strength / 10
    return strengthFactor
end

function RoadCraftTrailer.streamWriteGroundType(streamId, groundType)
    streamWriteUIntN(streamId, groundType, RoadCraftTrailer.NUM_BITS)
end

function RoadCraftTrailer.streamReadGroundType(streamId)
    local groundType = streamReadUIntN(streamId, RoadCraftTrailer.NUM_BITS)
    return groundType
end

function RoadCraftTrailer:processWorkAreaPlaceholder(workArea, dt)
    Logging.info(modName..": processWorkAreaPlaceholder", modName)
end

addModEventListener(RoadCraftTrailer)